-- ============================================
-- 0. 初期設定
-- ============================================
DROP TABLE IF EXISTS products;

-- ============================================
-- 1. テーブル作成
-- ============================================
CREATE TABLE products (
  id INT PRIMARY KEY,             -- 商品ID（手動入力）
  name VARCHAR(100) NOT NULL,     -- 商品名
  category VARCHAR(50) NOT NULL,  -- カテゴリ（ドリンク・フード・デザート・軽食など）
  price INT NOT NULL,             -- 価格
  stock INT NOT NULL,             -- 在庫数
  reg_date DATE                   -- 登録日
);

-- ============================================
-- 2. 初期データ挿入
-- ============================================
INSERT INTO products (id, name, category, price, stock, reg_date) VALUES
(1, 'ブレンドコーヒー', 'ドリンク', 450, 30, '2019-02-01'),
(2, 'アイスコーヒー', 'ドリンク', 500, 20, '2020-06-15'),
(3, 'カフェラテ', 'ドリンク', 550, 25, '2021-09-10'),
(4, 'コーヒーカップ', 'グッズ', 1200, 15, '2024-01-05'),
(5, 'クロワッサン', '軽食', 300, 15, '2023-03-05'),
(6, 'タンブラー', 'グッズ', 1500, 10, '2023-12-10'),
(7, 'チーズケーキ', 'デザート', 600, 8, '2020-10-30'),
(8, 'ショートケーキ', 'デザート', 650, 5, '2021-05-14'),
(9, 'モンブラン', 'デザート', 700, 6, '2023-11-22'),
(10, 'アップルパイ', 'デザート', 550, 9, '2019-08-18'),
(11, 'ステンレスボトル', 'グッズ', 2500, 12, '2024-02-01'),
(12, 'ホットサンド', '軽食', 700, 10, '2022-07-20'),
(13, 'エコバッグ', 'グッズ', 800, 0, '2023-11-25'),
(14, '紅茶', 'ドリンク', 400, 15, '2019-12-25')
;

-- ============================================
-- 3. SQL実行例（各章の学習内容）
-- ============================================

-- 1-1 全カラムSELECT
SELECT
  *
FROM
  products
;

-- 1-2 特定のカラムをSELECT
SELECT
  name,
  price
FROM
  products
;

-- 1-3 DISTINCTを使用（重複なく抽出）
SELECT DISTINCT
  category
FROM
  products
;

-- 1-4 WHERE 一致条件で抽出
SELECT
  *
FROM
  products
WHERE
  category = 'デザート'
;

-- 1-5 WHERE 曖昧検索
SELECT
  *
FROM
  products
WHERE
  name LIKE '%コーヒー%'
;

-- 1-6 WHERE句にANDを使用
SELECT
  *
FROM
  products
WHERE
  price >= 600
  AND category = 'デザート'
;

-- 1-7 WHERE句にORを使用
SELECT
  *
FROM
  products
WHERE
  category = '軽食'
  OR price >= 600
;

-- 1-8 ORDER BY句を使用（価格が安い順）
SELECT
  *
FROM
  products
ORDER BY
  price ASC
;

-- 1-9 LIMIT句を使用（上位3件のみ取得）
SELECT
  *
FROM
  products
ORDER BY
  stock DESC
LIMIT
  0, 3
;

-- 1-10 GROUP BY句を使用
SELECT
  category,
  COUNT(*)
FROM
  products
GROUP BY
  category
;

-- 1-11 AS（カラムへの別名）を使用（価格のカラム名を「値段」に変更）
SELECT
  name,
  price AS 値段
FROM
  products
;

-- 1-12 文字列／日付関数を使用（日付を日付文字列に変換）
SELECT
  name,
  DATE_FORMAT(reg_date, '%Y年%m月%d日')
FROM
  products
;

-- 2-1 INSERT（1行）
INSERT INTO products
(
  id,
  name,
  category,
  price,
  stock,
  reg_date
) VALUES
(
  15,
  'エスプレッソ',
  'ドリンク',
  480,
  10,
  '2025-04-01'
);


-- 2-2 INSERT（複数行）
INSERT INTO products
(
  id,
  name,
  category,
  price,
  stock,
  reg_date
) VALUES
(
  16,
  'フレンチトースト',
  '軽食',
  600,
  7,
  '2025-06-02'
),
(
  17,
  'パンケーキ',
  'デザート',
  650,
  8,
  '2025-06-23'
);

-- 3-1 UPDATE（Where句なし） ※全商品の価格を10%アップ
UPDATE
  products
SET
  price = price * 1.10
;

-- 3-2 UPDATE（Where句あり） ※「ブレンドコーヒー」の価格を500円に変更
UPDATE
  products
SET
  price = 700
WHERE
  name = 'カフェラテ'
;

-- 4-1 DELETE（Where句あり） ※在庫が0の商品を削除
DELETE FROM
  products
WHERE
  stock = 0
;

-- 4-2 DELETE（Where句なし） ※すべてのデータを削除（慎重に実行！）
-- DELETE FROM products;
